#version 130
#extension GL_EXT_gpu_shader4 : enable
// the version and open GL extension
// should be the first line of the shader
/////////////////////////////////////////////////////////////////////////////////
// Balloon TripMod01.fsh  by  ianthehenry  
//https://www.shadertoy.com/view/fl3fWS
//Licence : Creative Commons Attribution-ShareAlike 4.0
//http://creativecommons.org/licences/by-sa/4.0
// Adapted, trivialy, for use in VGHD player
/////////////////////////////////////////////
uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

#define iTime u_Elapsed*0.314159  //*0.1666
#define iResolution u_WindowSize

//#define mouse AUTO_MOUSE
//#define MOUSE_SPEED vec2(vec2(0.5,0.577777) * 0.25)
//#define MOUSE_POS   vec2((1.0+cos(iTime*MOUSE_SPEED))*u_WindowSize/2.0)
//#define MOUSE_PRESS vec2(0.0,0.0)
//#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )
//#define RIGID_SCROLL
// alternatively use static mouse definition
#define iMouse vec4(0.0,0.0, 0.0,0.0)
//#define iMouse vec4(512,256,180,120)
uniform sampler2D texture0;
uniform sampler2D texture1;
uniform sampler2D texture2;
uniform sampler2D texture3;
vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}
#define texture2D texture2D_Fract

const int MAX_STEPS = 256;
const float MINIMUM_HIT_DISTANCE = 0.1;
const float NORMAL_OFFSET = 0.005;
const float MAXIMUM_TRACE_DISTANCE = 64.0 * 1024.0;

/*

I created this using https://bauble.studio, although this code has
been formatted to fit your TV.

I recorded a video of myself modeling the balloon (the animation
here is a pretty simple modification):

https://www.youtube.com/watch?v=0-OtdjiR7dc&list=PLjT5GDnW_UMBS6ih0kG7jWB0n1SnotnEu

This is the original Bauble source for the image:

    (def bottom (+ -100 -50))

    (defn quantize [x y]
      (x * y | round / y))

    (def envelope
      (sphere 100
      | union :r 50 (cylinder :y 25 50 | move :y -100)
      | scale
         :y (ss p.y bottom 100 1 0.80)
         :z (ss p.y bottom (+ bottom 75) 1 0.50)
      | onion 0.1
      | subtract :r 1 (half-space :-y (+ bottom 7))
      | radial :y 18 (fn [$ i]
        (resurface $
          (shade (hsv (/ p.y 100 - (* 0.2 t) | quantize 8) 1 1)
          | move :y (sin (t * 1.5) | ss * 100 / 8 * (mod i 2))
          | rotate :x (sin (t / 2) * pi/4)
          | rotate :y (cos+ t)
          )
        | scale :z (mod i 2 * 2 - 1)))
      | color (* c (ss occlusion 0 0.75 0 1))
      | fresnel))

    (def flame
      (cone :-y 50 403
      | union :r 10 (cone :y 31 148)
      | bounded-offset 30 :threshold 3
        (perlin (- (* p [0.05 0.025 0.05]) [0 (* 5 t) 0]))
      | color (+ (* [1 1 0.7] (ss p.y -72 400))
        (mix [1 0.25 0] [1 1 0] (- normal.y)))
      | move :y -72))


    (def metal
      (shade [0.37 0.25 0.21] :gloss 7 :shine 1
      | fresnel [0.5 0.5 1]))

    (def torch
      (sphere 20
      | scale :y 0.75
      | union :r 3 (cylinder :r 1 :y (+ 20 (sin (p.y * 2))) 2 | move :y 10)
      | subtract (sphere 20 | move :y 20)
      | union :r 1 (line [30 10 0] [0 -10 0] 0.5 | union :r 1 (torus :x 2 0.5 | move [30 12 0]) | radial :y 6)
      | resurface metal
      ))

    (def basket
      (box 50
      | onion 10
      | subtract :r 10
        (half-space :y (+ 29 (pow (p.xz / 15) 2 | sum)))
      | offset (* 0.25 (+
        (cos (min (abs p.xz) * 3 + (step (sin (* 2 p.y)) 0 * pi/2)) | abs)
        (sin (p.y * 1) | abs)))
      | shade (hsv 0.10 0.54 0.88)
      | union
        (box [66 5 10] :r 5
        | move [0 (+ 48 (pow (p.x / 15) 2)) 55]
        | rotate :y (- tau/8)
        | mirror :x :z :r 10
        | rotate :y tau/8
        | shade [0.4 0.05 0.1] :gloss 3 :shine 0.5
        | fresnel)))

    (def collar
      (cylinder :y 50 8 :r 2
      | subtract :r 4 (cylinder :y 46 100)
      | union :r 1
        (union :r 1
          (cylinder :r 1 :x 4 4 | move :x 48)
          (cone :r 1 :x 2 2 | move :x 51)
        | radial :y 12)
      | rotate :y tau/8
      | resurface metal
      ))

    (def basket-corner [10 -38 10])
    (def basket-ropes
      (union
        (line [24 4 6] basket-corner 0.15)
        (line [17 4 17] basket-corner 0.15)
        (line [6 4 24] basket-corner 0.15)
      | mirror :x :z
      | offset (sin+ (p.y * 20) * 0.01)
      | shade (hsv 0.1 0.9 0.5)
      | fresnel 0.05 :exponent 2
      ))

    (def torch-ropes
      (line [24 4 6] [9 -10.25 0] 0.1
      | mirror :z
      | radial :y 6
      | resurface metal))

    (union
      (envelope
      | color (+ c (* 0.05 [1 1 0]
        (ss (distance p [0 bottom 0]) 0 150 1 0)
        (perlin [(* 2 t) 0]))))
      (bound (union
        (flame | scale 0.1 | move :y (- bottom 5))
        (torch | scale 0.3 | move :y (- bottom 14))
        (basket | scale 0.2 | move :y (- bottom 50))
        (collar | scale 0.53 | move :y (- bottom -5))
        (basket-ropes | move :y bottom)
        (torch-ropes | move :y bottom))
      (sphere 45 | move :y (- bottom 25)))
    | rotate :y (sin t * 1.5 + t * 0.1)
    | move :y 81)

*/

const float PI = 3.14159265359;
const float DEG_TO_RAD = PI / 180.0;

struct Light {
  vec3 position;
  vec3 color;
  float radius;
};

const Light lights[1] = Light[1](
  Light(vec3(512.0, 512.0, 512.0), vec3(1.0), 4096.0)
);
float s3d_cylinder_0(vec3 p, float radius, float height) {
  vec2 d = abs(vec2(length(p.xz), p.y)) - vec2(radius, height);
  return min(max(d.x, d.y), 0.0) + length(max(d, 0.0));

}
float union_0(vec3 p, float r) {
  float b, h = 0.0;
  r = max(r, 0.0000000001);
  float a = (length(p) - 100.0);
  float _r1; {
    vec3 p1 = (p - vec3(0.0, -100.0, 0.0));
    _r1 = s3d_cylinder_0(p1, 25.0, 50.0);
  }
  b = _r1;
  h = clamp(0.5 + 0.5 * (b - a) / r, 0.0, 1.0);
  a = mix(b, a, h) - r * h * (1.0 - h);

  return a;
}
float min3(vec3 p) {
  return min(p.x, min(p.y, p.z));
}
float subtract_0(vec3 p, float r) {
  float b, h = 0.0;
  r = max(r, 0.0000000001);
  float _r1; {
    vec3 scale0_1 = vec3(1.0, ((1.0 * (1.0 + (smoothstep(-150.0, 100.0, p.y) * -0.200000))) * 1.0), (1.0 * (1.0 + (smoothstep(-150.0, -75.0, p.y) * -0.500000))));
    vec3 p1 = (p / scale0_1);
    _r1 = (abs((union_0(p1, 50.0) * min3(abs(scale0_1)))) - 0.100000);
  }
  float a = _r1;
  float _r2; {
    vec3 p1 = (p - vec3(0.0, -143.0, 0.0));
    _r2 = p1.y;
  }
  b = _r2;
  h = clamp(0.5 - 0.5 * (a + b) / r, 0.0, 1.0);
  a = mix(a, -b, h) + r * h * (1.0 - h);

  return a;
}
float atan2(float y, float x) {
  return x == 0.0 ? sign(y) * 0.5 * PI : atan(y, x);

}
mat3 rotate_y(float angle) {
  float s = sin(angle);
  float c = cos(angle);
  return mat3(
    c, 0.0, s,
    0.0, 1.0, 0.0,
    -s, 0.0, c);
}
float s3d_cone_0(vec3 p, float radius, float height) {
  vec2 q = vec2(radius, height);
  vec2 w = vec2(length(p.xz), p.y);
  vec2 a = w - q * clamp(dot(w, q) / dot(q, q), 0.0, 1.0);
  vec2 b = w - q * vec2(clamp(w.x / q.x, 0.0, 1.0), 1.0);
  float k = sign(q.y);
  float d = min(dot(a, a), dot(b, b));
  float s = max(k * (w.x * q.y - w.y * q.x), k * (w.y - q.y));
  return sqrt(d) * sign(s);

}
float s3d_cone_1(vec3 p, float radius, float height) {
  vec2 q = vec2(radius, height);
  vec2 w = vec2(length(p.xz), height - p.y);
  vec2 a = w - q * clamp(dot(w, q) / dot(q, q), 0.0, 1.0);
  vec2 b = w - q * vec2(clamp(w.x / q.x, 0.0, 1.0), 1.0);
  float k = sign(q.y);
  float d = min(dot(a, a), dot(b, b));
  float s = max(k * (w.x * q.y - w.y * q.x), k * (w.y - q.y));
  return sqrt(d) * sign(s);

}
float union_1(vec3 p, float r) {
  float b, h = 0.0;
  r = max(r, 0.0000000001);
  float a = s3d_cone_0(p, 50.0, 403.0);
  b = s3d_cone_1(p, 31.0, 148.0);
  h = clamp(0.5 + 0.5 * (b - a) / r, 0.0, 1.0);
  a = mix(b, a, h) - r * h * (1.0 - h);

  return a;
}
vec3 fade3(vec3 t) {
  return t * t * t * (t * (t * 6.0 - 15.0) + 10.0);
}
vec4 taylor_inv_sqrt4(vec4 r) {
  return 1.79284291400159 - 0.85373472095314 * r;
}
vec3 mod289_3(vec3 x) {
  return x - floor(x * (1.0 / 289.0)) * 289.0;
}
vec4 mod289_4(vec4 x) {
  return x - floor(x * (1.0 / 289.0)) * 289.0;
}
vec4 permute4(vec4 x) {
  return mod289_4(((x * 34.0) + 10.0) * x);
}
// Perlin noise functions by Stefan Gustavson
// https://github.com/stegu/webgl-noise
float perlin3(vec3 P) {
  vec3 Pi0 = floor(P); // Integer part for indexing
  vec3 Pi1 = Pi0 + vec3(1.0); // Integer part + 1
  Pi0 = mod289_3(Pi0);
  Pi1 = mod289_3(Pi1);
  vec3 Pf0 = fract(P); // Fractional part for interpolation
  vec3 Pf1 = Pf0 - vec3(1.0); // Fractional part - 1.0
  vec4 ix = vec4(Pi0.x, Pi1.x, Pi0.x, Pi1.x);
  vec4 iy = vec4(Pi0.yy, Pi1.yy);
  vec4 iz0 = Pi0.zzzz;
  vec4 iz1 = Pi1.zzzz;

  vec4 ixy = permute4(permute4(ix) + iy);
  vec4 ixy0 = permute4(ixy + iz0);
  vec4 ixy1 = permute4(ixy + iz1);

  vec4 gx0 = ixy0 * (1.0 / 7.0);
  vec4 gy0 = fract(floor(gx0) * (1.0 / 7.0)) - 0.5;
  gx0 = fract(gx0);
  vec4 gz0 = vec4(0.5) - abs(gx0) - abs(gy0);
  vec4 sz0 = step(gz0, vec4(0.0));
  gx0 -= sz0 * (step(0.0, gx0) - 0.5);
  gy0 -= sz0 * (step(0.0, gy0) - 0.5);

  vec4 gx1 = ixy1 * (1.0 / 7.0);
  vec4 gy1 = fract(floor(gx1) * (1.0 / 7.0)) - 0.5;
  gx1 = fract(gx1);
  vec4 gz1 = vec4(0.5) - abs(gx1) - abs(gy1);
  vec4 sz1 = step(gz1, vec4(0.0));
  gx1 -= sz1 * (step(0.0, gx1) - 0.5);
  gy1 -= sz1 * (step(0.0, gy1) - 0.5);

  vec3 g000 = vec3(gx0.x, gy0.x, gz0.x);
  vec3 g100 = vec3(gx0.y, gy0.y, gz0.y);
  vec3 g010 = vec3(gx0.z, gy0.z, gz0.z);
  vec3 g110 = vec3(gx0.w, gy0.w, gz0.w);
  vec3 g001 = vec3(gx1.x, gy1.x, gz1.x);
  vec3 g101 = vec3(gx1.y, gy1.y, gz1.y);
  vec3 g011 = vec3(gx1.z, gy1.z, gz1.z);
  vec3 g111 = vec3(gx1.w, gy1.w, gz1.w);

  vec4 norm0 = taylor_inv_sqrt4(vec4(dot(g000, g000), dot(g010, g010), dot(g100, g100), dot(g110, g110)));
  g000 *= norm0.x;
  g010 *= norm0.y;
  g100 *= norm0.z;
  g110 *= norm0.w;
  vec4 norm1 = taylor_inv_sqrt4(vec4(dot(g001, g001), dot(g011, g011), dot(g101, g101), dot(g111, g111)));
  g001 *= norm1.x;
  g011 *= norm1.y;
  g101 *= norm1.z;
  g111 *= norm1.w;

  float n000 = dot(g000, Pf0);
  float n100 = dot(g100, vec3(Pf1.x, Pf0.yz));
  float n010 = dot(g010, vec3(Pf0.x, Pf1.y, Pf0.z));
  float n110 = dot(g110, vec3(Pf1.xy, Pf0.z));
  float n001 = dot(g001, vec3(Pf0.xy, Pf1.z));
  float n101 = dot(g101, vec3(Pf1.x, Pf0.y, Pf1.z));
  float n011 = dot(g011, vec3(Pf0.x, Pf1.yz));
  float n111 = dot(g111, Pf1);

  vec3 fade_xyz = fade3(Pf0);
  vec4 n_z = mix(vec4(n000, n100, n010, n110), vec4(n001, n101, n011, n111), fade_xyz.z);
  vec2 n_yz = mix(n_z.xy, n_z.zw, fade_xyz.y);
  float n_xyz = mix(n_yz.x, n_yz.y, fade_xyz.x);
  return 2.2 * n_xyz;

}
float bound_0(float magnitude0_, float t, vec3 p, float threshold) {
  float boundary_distance = (union_1(p, 10.0) - magnitude0_);
  if (boundary_distance < threshold) {
    return (union_1(p, 10.0) - (magnitude0_ * perlin3(((p * vec3(0.050000, 0.025000, 0.050000)) - vec3(0.0, (5.0 * t), 0.0)))));
  } else {
    return boundary_distance;
  }
}
float union_2(vec3 p, float r) {
  float b, h = 0.0;
  r = max(r, 0.0000000001);
  float _r1; {
    vec3 scale0_1 = vec3(1.0, 0.750000, 1.0);
    vec3 p1 = (p / scale0_1);
    _r1 = ((length(p1) - 20.0) * min3(abs(scale0_1)));
  }
  float a = _r1;
  float _r2; {
    vec3 p1 = (p - vec3(0.0, 10.0, 0.0));
    _r2 = (s3d_cylinder_0(p1, ((20.0 + sin((p1.y * 2.0))) - 1.0), 1.0) - 1.0);
  }
  b = _r2;
  h = clamp(0.5 + 0.5 * (b - a) / r, 0.0, 1.0);
  a = mix(b, a, h) - r * h * (1.0 - h);

  return a;
}
float subtract_1(vec3 p) {
  float d = union_2(p, 3.0);
  float _r1; {
    vec3 p1 = (p - vec3(0.0, 20.0, 0.0));
    _r1 = (length(p1) - 20.0);
  }
  d = max(d, -_r1);
  return d;
}
float s3d_line(vec3 p, vec3 a, vec3 b) {
  vec3 pa = p - a, ba = b - a;
  float h = clamp(dot(pa, ba) / dot(ba, ba), 0.0, 1.0);
  return length(pa - ba * h);

}
float s3d_torus_0(vec3 p, float major_radius, float minor_radius) {
  vec2 q = vec2(length(p.yz) - major_radius, p.x);
  return length(q) - minor_radius;

}
float union_3(vec3 p, float r) {
  float b, h = 0.0;
  r = max(r, 0.0000000001);
  float a = (s3d_line(p, vec3(30.0, 10.0, 0.0), vec3(0.0, -10.0, 0.0)) - 0.500000);
  float _r1; {
    vec3 p1 = (p - vec3(30.0, 12.0, 0.0));
    _r1 = s3d_torus_0(p1, 2.0, 0.500000);
  }
  b = _r1;
  h = clamp(0.5 + 0.5 * (b - a) / r, 0.0, 1.0);
  a = mix(b, a, h) - r * h * (1.0 - h);

  return a;
}
float union_4(vec3 p, float r) {
  float b, h = 0.0;
  r = max(r, 0.0000000001);
  float a = subtract_1(p);
  float _r1; {
    float angle0_1 = 1.047198;
    float index0_1 = floor((mod((atan2(p.z, p.x) + (0.500000 * angle0_1)), 6.283185) / angle0_1));
    vec3 p1 = ((p * rotate_y((index0_1 * angle0_1))) - vec3(0.0, 0.0, 0.0));
    _r1 = union_3(p1, 1.0);
  }
  b = _r1;
  h = clamp(0.5 + 0.5 * (b - a) / r, 0.0, 1.0);
  a = mix(b, a, h) - r * h * (1.0 - h);

  return a;
}
float s3d_box(vec3 p, vec3 size) {
  vec3 q = abs(p) - size;
  return length(max(q, 0.0)) + min(max(q.x, max(q.y, q.z)), 0.0);

}
float sum2(vec2 p) {
  return p.x + p.y;
}
float subtract_2(vec3 p, float r) {
  float b, h = 0.0;
  r = max(r, 0.0000000001);
  float a = (abs(s3d_box(p, vec3(50.0, 50.0, 50.0))) - 10.0);
  float _r1; {
    vec3 p1 = (p - vec3(0.0, (29.0 + sum2(pow((p.xz / 15.0), vec2(2.0)))), 0.0));
    _r1 = (-p1.y);
  }
  b = _r1;
  h = clamp(0.5 - 0.5 * (a + b) / r, 0.0, 1.0);
  a = mix(a, -b, h) + r * h * (1.0 - h);

  return a;
}
float min2(vec2 p) {
  return min(p.x, p.y);
}
vec3 biased_sqrt_xz_0(vec3 p, float r) {
  p.xz = sqrt(p.xz * p.xz + r * r);
  return p;
}
float union_5(vec3 p) {
  float d = (subtract_2(p, 10.0) - (0.250000 * (abs(cos(((min2(abs(p.xz)) * 3.0) + (step(0.0, sin((2.0 * p.y))) * 1.570796)))) + abs(sin((p.y * 1.0))))));
  float _r1; {
    vec3 p1 = (p * mat3(0.707107, 0.0, 0.707107, 0.0, 1.0, 0.0, -0.707107, 0.0, 0.707107));
    vec3 p2 = biased_sqrt_xz_0(p1, 10.0);
    vec3 p3 = (p2 * mat3(0.707107, 0.0, -0.707107, 0.0, 1.0, 0.0, 0.707107, 0.0, 0.707107));
    vec3 p4 = (p3 - vec3(0.0, (0.0 + (48.0 + pow((p3.x / 15.0), 2.0))), 55.0));
    _r1 = (s3d_box(p4, vec3(61.0, 0.0, 5.0)) - 5.0);
  }
  d = min(d, _r1);
  return d;
}
float subtract_3(vec3 p, float r) {
  float b, h = 0.0;
  r = max(r, 0.0000000001);
  float a = (s3d_cylinder_0(p, 48.0, 6.0) - 2.0);
  b = s3d_cylinder_0(p, 46.0, 100.0);
  h = clamp(0.5 - 0.5 * (a + b) / r, 0.0, 1.0);
  a = mix(a, -b, h) + r * h * (1.0 - h);

  return a;
}
float s3d_cylinder_1(vec3 p, float radius, float height) {
  vec2 d = abs(vec2(length(p.yz), p.x)) - vec2(radius, height);
  return min(max(d.x, d.y), 0.0) + length(max(d, 0.0));

}
float s3d_cone_2(vec3 p, float radius, float height) {
  vec2 q = vec2(radius, height);
  vec2 w = vec2(length(p.yz), height - p.x);
  vec2 a = w - q * clamp(dot(w, q) / dot(q, q), 0.0, 1.0);
  vec2 b = w - q * vec2(clamp(w.x / q.x, 0.0, 1.0), 1.0);
  float k = sign(q.y);
  float d = min(dot(a, a), dot(b, b));
  float s = max(k * (w.x * q.y - w.y * q.x), k * (w.y - q.y));
  return sqrt(d) * sign(s);

}
float union_6(vec3 p, float r) {
  float b, h = 0.0;
  r = max(r, 0.0000000001);
  float _r1; {
    vec3 p1 = (p - vec3(48.0, 0.0, 0.0));
    _r1 = (s3d_cylinder_1(p1, 3.0, 3.0) - 1.0);
  }
  float a = _r1;
  float _r2; {
    vec3 p1 = (p - vec3(51.0, 0.0, 0.0));
    float height0_1 = 2.0;
    float radius0_1 = 2.0;
    float r0_1 = 1.0;
    float tip_offset0_1 = (r0_1 * (height0_1 / radius0_1));
    vec3 p2 = (p1 - vec3(r0_1, 0.0, 0.0));
    _r2 = (s3d_cone_2(p2, (radius0_1 - r0_1), (height0_1 - tip_offset0_1)) - r0_1);
  }
  b = _r2;
  h = clamp(0.5 + 0.5 * (b - a) / r, 0.0, 1.0);
  a = mix(b, a, h) - r * h * (1.0 - h);

  return a;
}
float union_7(vec3 p, float r) {
  float b, h = 0.0;
  r = max(r, 0.0000000001);
  float a = subtract_3(p, 4.0);
  float _r1; {
    float angle0_1 = 0.523599;
    float index0_1 = floor((mod((atan2(p.z, p.x) + (0.500000 * angle0_1)), 6.283185) / angle0_1));
    vec3 p1 = ((p * rotate_y((index0_1 * angle0_1))) - vec3(0.0, 0.0, 0.0));
    _r1 = union_6(p1, 1.0);
  }
  b = _r1;
  h = clamp(0.5 + 0.5 * (b - a) / r, 0.0, 1.0);
  a = mix(b, a, h) - r * h * (1.0 - h);

  return a;
}
vec3 abs_xz_0(vec3 p) {
  p.xz = abs(p.xz);
  return p;
}
float union_8(vec3 p) {
  float d = (s3d_line(p, vec3(24.0, 4.0, 6.0), vec3(10.0, -38.0, 10.0)) - 0.150000);
  d = min(d, (s3d_line(p, vec3(17.0, 4.0, 17.0), vec3(10.0, -38.0, 10.0)) - 0.150000));
  d = min(d, (s3d_line(p, vec3(6.0, 4.0, 24.0), vec3(10.0, -38.0, 10.0)) - 0.150000));
  return d;
}
vec3 abs_z_0(vec3 p) {
  p.z = abs(p.z);
  return p;
}
float union_9(float t, vec3 p) {
  float _r1; {
    vec3 p1 = (p - vec3(0.0, -155.0, 0.0));
    float scale0_1 = 0.100000;
    vec3 p2 = (p1 / scale0_1);
    vec3 p3 = (p2 - vec3(0.0, -72.0, 0.0));
    float magnitude0_1 = 30.0;
    _r1 = (bound_0(magnitude0_1, t, p3, 3.0) * scale0_1);
  }
  float d = _r1;
  float _r2; {
    vec3 p1 = (p - vec3(0.0, -164.0, 0.0));
    float scale0_1 = 0.300000;
    vec3 p2 = (p1 / scale0_1);
    _r2 = (union_4(p2, 1.0) * scale0_1);
  }
  d = min(d, _r2);
  float _r3; {
    vec3 p1 = (p - vec3(0.0, -200.0, 0.0));
    float scale0_1 = 0.200000;
    vec3 p2 = (p1 / scale0_1);
    _r3 = (union_5(p2) * scale0_1);
  }
  d = min(d, _r3);
  float _r4; {
    vec3 p1 = (p - vec3(0.0, -145.0, 0.0));
    float scale0_1 = 0.530000;
    vec3 p2 = (p1 / scale0_1);
    vec3 p3 = (p2 * mat3(0.707107, 0.0, 0.707107, 0.0, 1.0, 0.0, -0.707107, 0.0, 0.707107));
    _r4 = (union_7(p3, 1.0) * scale0_1);
  }
  d = min(d, _r4);
  float _r5; {
    vec3 p1 = (p - vec3(0.0, -150.0, 0.0));
    vec3 p2 = abs_xz_0(p1);
    _r5 = (union_8(p2) - ((0.500000 * (sin((p1.y * 20.0)) + 1.0)) * 0.010000));
  }
  d = min(d, _r5);
  float _r6; {
    vec3 p1 = (p - vec3(0.0, -150.0, 0.0));
    float angle0_1 = 1.047198;
    float index0_1 = floor((mod((atan2(p1.z, p1.x) + (0.500000 * angle0_1)), 6.283185) / angle0_1));
    vec3 p2 = ((p1 * rotate_y((index0_1 * angle0_1))) - vec3(0.0, 0.0, 0.0));
    vec3 p3 = abs_z_0(p2);
    _r6 = (s3d_line(p3, vec3(24.0, 4.0, 6.0), vec3(9.0, -10.250000, 0.0)) - 0.100000);
  }
  d = min(d, _r6);
  return d;
}
float bound_1(float t, vec3 p, float threshold) {
  vec3 p1 = (p - vec3(0.0, -175.0, 0.0));
  float boundary_distance = (length(p1) - 45.0);
  if (boundary_distance < threshold) {
    return union_9(t, p);
  } else {
    return boundary_distance;
  }
}
float union_10(float t, vec3 p) {
  float _r1; {
    float angle0_1 = 0.349066;
    float index0_1 = floor((mod((atan2(p.z, p.x) + (0.500000 * angle0_1)), 6.283185) / angle0_1));
    vec3 p1 = ((p * rotate_y((index0_1 * angle0_1))) - vec3(0.0, 0.0, 0.0));
    vec3 scale0_1 = vec3(1.0, 1.0, (1.0 * ((mod(index0_1, 2.0) * 2.0) - 1.0)));
    vec3 p2 = (p1 / scale0_1);
    _r1 = (subtract_0(p2, 1.0) * min3(abs(scale0_1)));
  }
  float d = _r1;
  d = min(d, bound_1(t, p, 1.0));
  return d;
}
float bound_2(float magnitude0_, float t, vec3 p, float threshold) {
  float boundary_distance = (union_1(p, 10.0) - magnitude0_);
  if (boundary_distance < threshold) {
    return (union_1(p, 10.0) - (magnitude0_ * perlin3(((p * vec3(0.050000, 0.025000, 0.050000)) - vec3(0.0, (5.0 * t), 0.0)))));
  } else {
    return boundary_distance;
  }
}
vec3 fresnel(vec3 normal, vec3 P, vec3 camera_origin, vec3 color, float strength, float exponent) {
  vec3 view_dir = normalize(camera_origin - P);
  float fresnel = pow(1.0 - dot(normal, view_dir), exponent);
  return color * strength * fresnel;

}
vec3 blinn_phong(vec3 normal, float occlusion, vec3 P, float light_intensities[1], vec3 camera_origin, vec3 color, float shine, float gloss, float ambient) {
  vec3 view_dir = normalize(camera_origin - P);
  vec3 result = color * ambient;
  for (int i = 0; i < lights.length(); i++) {
    vec3 light_color = lights[i].color * light_intensities[i];
    vec3 light_dir = normalize(lights[i].position - P);
    vec3 halfway_dir = normalize(light_dir + view_dir);
    float specular_strength = shine * pow(max(dot(normal, halfway_dir), 0.0), gloss * gloss);
    float diffuse = max(0.0, dot(normal, light_dir));
    result += light_color * specular_strength;
    result += color * diffuse * light_color;
  }
  return result;

}
mat3 rotate_x(float angle) {
  float s = sin(angle);
  float c = cos(angle);
  return mat3(
    1.0, 0.0, 0.0,
    0.0, c, -s,
    0.0, s, c);
}
vec3 hsv(float h, float s, float v) {
  vec3 p = abs(fract(vec3(h, h, h) + vec3(1.0, 2.0 / 3.0, 1.0 / 3.0)) * 6.0 - 3.0);
  return v * mix(vec3(1.0), clamp(p - 1.0, 0.0, 1.0), s);

}
vec2 fade2(vec2 t) {
  return t * t * t * (t * (t * 6.0 - 15.0) + 10.0);
}
// Perlin noise functions by Stefan Gustavson
// https://github.com/stegu/webgl-noise
float perlin2(vec2 P) {
  vec4 Pi = floor(P.xyxy) + vec4(0.0, 0.0, 1.0, 1.0);
  vec4 Pf = fract(P.xyxy) - vec4(0.0, 0.0, 1.0, 1.0);
  Pi = mod289_4(Pi); // To avoid truncation effects in permutation
  vec4 ix = Pi.xzxz;
  vec4 iy = Pi.yyww;
  vec4 fx = Pf.xzxz;
  vec4 fy = Pf.yyww;

  vec4 i = permute4(permute4(ix) + iy);

  vec4 gx = fract(i * (1.0 / 41.0)) * 2.0 - 1.0;
  vec4 gy = abs(gx) - 0.5;
  vec4 tx = floor(gx + 0.5);
  gx = gx - tx;

  vec2 g00 = vec2(gx.x, gy.x);
  vec2 g10 = vec2(gx.y, gy.y);
  vec2 g01 = vec2(gx.z, gy.z);
  vec2 g11 = vec2(gx.w, gy.w);

  vec4 norm = taylor_inv_sqrt4(vec4(dot(g00, g00), dot(g01, g01), dot(g10, g10), dot(g11, g11)));
  g00 *= norm.x;
  g01 *= norm.y;
  g10 *= norm.z;
  g11 *= norm.w;

  float n00 = dot(g00, vec2(fx.x, fy.x));
  float n10 = dot(g10, vec2(fx.y, fy.y));
  float n01 = dot(g01, vec2(fx.z, fy.z));
  float n11 = dot(g11, vec2(fx.w, fy.w));

  vec2 fade_xy = fade2(Pf.xy);
  vec2 n_x = mix(vec2(n00, n01), vec2(n10, n11), fade_xy.x);
  float n_xy = mix(n_x.x, n_x.y, fade_xy.y);
  return 2.3 * n_xy;
}
float bound_3(float magnitude0_, float t, vec3 p, float threshold) {
  float boundary_distance = (union_1(p, 10.0) - magnitude0_);
  if (boundary_distance < threshold) {
    return (union_1(p, 10.0) - (magnitude0_ * perlin3(((p * vec3(0.050000, 0.025000, 0.050000)) - vec3(0.0, (5.0 * t), 0.0)))));
  } else {
    return boundary_distance;
  }
}
vec3 union_color_0(vec3 normal, vec3 p, float r) {
  float b, h;
  r = max(r, 0.0000000001);
  float a = s3d_cone_0(p, 50.0, 403.0);
  vec3 color = (0.500000 * (1.0 + normal));
  b = s3d_cone_1(p, 31.0, 148.0);
  h = clamp(0.5 + 0.5 * (b - a) / r, 0.0, 1.0);
  a = mix(b, a, h) - r * h * (1.0 - h);
  color = mix((0.500000 * (1.0 + normal)), color, h);
  return color;
}
vec3 union_color_1(vec3 p, vec3 normal, float occlusion, vec3 P, float light_intensities[1], vec3 camera_origin) {
  float d = (subtract_2(p, 10.0) - (0.250000 * (abs(cos(((min2(abs(p.xz)) * 3.0) + (step(0.0, sin((2.0 * p.y))) * 1.570796)))) + abs(sin((p.y * 1.0))))));
  float d2;
  int surface = 0;
  vec3 color;
  float _r1; {
    vec3 p1 = (p * mat3(0.707107, 0.0, 0.707107, 0.0, 1.0, 0.0, -0.707107, 0.0, 0.707107));
    vec3 p2 = biased_sqrt_xz_0(p1, 10.0);
    vec3 p3 = (p2 * mat3(0.707107, 0.0, -0.707107, 0.0, 1.0, 0.0, 0.707107, 0.0, 0.707107));
    vec3 p4 = (p3 - vec3(0.0, (0.0 + (48.0 + pow((p3.x / 15.0), 2.0))), 55.0));
    _r1 = (s3d_box(p4, vec3(61.0, 0.0, 5.0)) - 5.0);
  }
  d2 = _r1;
  if (d2 < d) {
    d = d2;
    surface = 1;
  }
  switch (surface) {
  case 0:
    color = blinn_phong(normal, occlusion, P, light_intensities, camera_origin, vec3(0.880000, 0.689920, 0.404800), 0.250000, 4.0, 0.200000);
    break;
  case 1:
    color = (blinn_phong(normal, occlusion, P, light_intensities, camera_origin, vec3(0.400000, 0.050000, 0.100000), 0.500000, 3.0, 0.200000) + fresnel(normal, P, camera_origin, vec3(1.0, 1.0, 1.0), 0.250000, 5.0));
    break;
  }
  return color;
}
vec3 union_color_2(vec3 p, vec3 normal, float occlusion, vec3 P, float t, float light_intensities[1], vec3 camera_origin) {
  float _r1; {
    vec3 p1 = (p - vec3(0.0, -155.0, 0.0));
    float scale0_1 = 0.100000;
    vec3 p2 = (p1 / scale0_1);
    vec3 p3 = (p2 - vec3(0.0, -72.0, 0.0));
    float magnitude0_1 = 30.0;
    _r1 = (bound_3(magnitude0_1, t, p3, 3.0) * scale0_1);
  }
  float d = _r1;
  float d2;
  int surface = 0;
  vec3 color;
  float _r2; {
    vec3 p1 = (p - vec3(0.0, -164.0, 0.0));
    float scale0_1 = 0.300000;
    vec3 p2 = (p1 / scale0_1);
    _r2 = (union_4(p2, 1.0) * scale0_1);
  }
  d2 = _r2;
  if (d2 < d) {
    d = d2;
    surface = 1;
  }
  float _r3; {
    vec3 p1 = (p - vec3(0.0, -200.0, 0.0));
    float scale0_1 = 0.200000;
    vec3 p2 = (p1 / scale0_1);
    _r3 = (union_5(p2) * scale0_1);
  }
  d2 = _r3;
  if (d2 < d) {
    d = d2;
    surface = 2;
  }
  float _r4; {
    vec3 p1 = (p - vec3(0.0, -145.0, 0.0));
    float scale0_1 = 0.530000;
    vec3 p2 = (p1 / scale0_1);
    vec3 p3 = (p2 * mat3(0.707107, 0.0, 0.707107, 0.0, 1.0, 0.0, -0.707107, 0.0, 0.707107));
    _r4 = (union_7(p3, 1.0) * scale0_1);
  }
  d2 = _r4;
  if (d2 < d) {
    d = d2;
    surface = 3;
  }
  float _r5; {
    vec3 p1 = (p - vec3(0.0, -150.0, 0.0));
    vec3 p2 = abs_xz_0(p1);
    _r5 = (union_8(p2) - ((0.500000 * (sin((p1.y * 20.0)) + 1.0)) * 0.010000));
  }
  d2 = _r5;
  if (d2 < d) {
    d = d2;
    surface = 4;
  }
  float _r6; {
    vec3 p1 = (p - vec3(0.0, -150.0, 0.0));
    float angle0_1 = 1.047198;
    float index0_1 = floor((mod((atan2(p1.z, p1.x) + (0.500000 * angle0_1)), 6.283185) / angle0_1));
    vec3 p2 = ((p1 * rotate_y((index0_1 * angle0_1))) - vec3(0.0, 0.0, 0.0));
    vec3 p3 = abs_z_0(p2);
    _r6 = (s3d_line(p3, vec3(24.0, 4.0, 6.0), vec3(9.0, -10.250000, 0.0)) - 0.100000);
  }
  d2 = _r6;
  if (d2 < d) {
    d = d2;
    surface = 5;
  }
  switch (surface) {
  case 0:
    vec3 _r7; {
      vec3 p1 = (p - vec3(0.0, -155.0, 0.0));
      vec3 p2 = (p1 / 0.100000);
      vec3 p3 = (p2 - vec3(0.0, -72.0, 0.0));
      vec3 color0_1 = union_color_0(normal, p3, 10.0);
      _r7 = ((vec3(1.0, 1.0, 0.700000) * smoothstep(-72.0, 400.0, p3.y)) + mix(vec3(1.0, 0.250000, 0.0), vec3(1.0, 1.0, 0.0), (-normal.y)));
    }
    color = _r7;
    break;
  case 1:
    vec3 _r8; {
      vec3 p1 = (p - vec3(0.0, -164.0, 0.0));
      vec3 p2 = (p1 / 0.300000);
      _r8 = (blinn_phong(normal, occlusion, P, light_intensities, camera_origin, vec3(0.370000, 0.250000, 0.210000), 1.0, 7.0, 0.200000) + fresnel(normal, P, camera_origin, vec3(0.500000, 0.500000, 1.0), 0.250000, 5.0));
    }
    color = _r8;
    break;
  case 2:
    vec3 _r9; {
      vec3 p1 = (p - vec3(0.0, -200.0, 0.0));
      vec3 p2 = (p1 / 0.200000);
      _r9 = union_color_1(p2, normal, occlusion, P, light_intensities, camera_origin);
    }
    color = _r9;
    break;
  case 3:
    vec3 _r10; {
      vec3 p1 = (p - vec3(0.0, -145.0, 0.0));
      vec3 p2 = (p1 / 0.530000);
      _r10 = (blinn_phong(normal, occlusion, P, light_intensities, camera_origin, vec3(0.370000, 0.250000, 0.210000), 1.0, 7.0, 0.200000) + fresnel(normal, P, camera_origin, vec3(0.500000, 0.500000, 1.0), 0.250000, 5.0));
    }
    color = _r10;
    break;
  case 4:
    vec3 _r11; {
      vec3 p1 = (p - vec3(0.0, -150.0, 0.0));
      _r11 = (blinn_phong(normal, occlusion, P, light_intensities, camera_origin, vec3(0.500000, 0.320000, 0.050000), 0.250000, 4.0, 0.200000) + fresnel(normal, P, camera_origin, vec3(1.0, 1.0, 1.0), 0.050000, 2.0));
    }
    color = _r11;
    break;
  case 5:
    vec3 _r12; {
      vec3 p1 = (p - vec3(0.0, -150.0, 0.0));
      _r12 = (blinn_phong(normal, occlusion, P, light_intensities, camera_origin, vec3(0.370000, 0.250000, 0.210000), 1.0, 7.0, 0.200000) + fresnel(normal, P, camera_origin, vec3(0.500000, 0.500000, 1.0), 0.250000, 5.0));
    }
    color = _r12;
    break;
  }
  return color;
}
vec3 union_color_3(vec3 p, vec3 normal, float occlusion, vec3 P, float t, float light_intensities[1], vec3 camera_origin) {
  float _r1; {
    float angle0_1 = 0.349066;
    float index0_1 = floor((mod((atan2(p.z, p.x) + (0.500000 * angle0_1)), 6.283185) / angle0_1));
    vec3 p1 = ((p * rotate_y((index0_1 * angle0_1))) - vec3(0.0, 0.0, 0.0));
    vec3 scale0_1 = vec3(1.0, 1.0, (1.0 * ((mod(index0_1, 2.0) * 2.0) - 1.0)));
    vec3 p2 = (p1 / scale0_1);
    _r1 = (subtract_0(p2, 1.0) * min3(abs(scale0_1)));
  }
  float d = _r1;
  float d2;
  int surface = 0;
  vec3 color;
  d2 = bound_1(t, p, 1.0);
  if (d2 < d) {
    d = d2;
    surface = 1;
  }
  switch (surface) {
  case 0:
    vec3 _r2; {
      float angle0_1 = 0.349066;
      float index0_1 = floor((mod((atan2(p.z, p.x) + (0.500000 * angle0_1)), 6.283185) / angle0_1));
      vec3 p1 = ((p * rotate_y((index0_1 * angle0_1))) - vec3(0.0, 0.0, 0.0));
      vec3 p2 = (p1 / vec3(1.0, 1.0, (1.0 * ((mod(index0_1, 2.0) * 2.0) - 1.0))));
      vec3 p3 = (p2 * (mat3(1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0) * rotate_y((1.0 * (0.500000 * (cos(t) + 1.0))))));
      vec3 p4 = (p3 * (mat3(1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0) * rotate_x((1.0 * (sin((t / 2.0)) * 0.785398)))));
      vec3 p5 = (p4 - vec3(0.0, (0.0 + (((smoothstep(0.0, 1.0, sin((t * 1.500000))) * 100.0) / 8.0) * mod(index0_1, 2.0))), 0.0));
      vec3 color1_1 = blinn_phong(normal, occlusion, P, light_intensities, camera_origin, hsv((round((((p5.y / 100.0) - (0.200000 * t)) * 8.0)) / 8.0), 1.0, 1.0), 0.250000, 4.0, 0.200000);
      vec3 color0_1 = ((color1_1 * (0.0 + (smoothstep(0.0, 0.750000, occlusion) * 1.0))) + fresnel(normal, P, camera_origin, vec3(1.0, 1.0, 1.0), 0.250000, 5.0));
      _r2 = (color0_1 + ((vec3(0.050000, 0.050000, 0.0) * (1.0 + (smoothstep(0.0, 150.0, distance(p, vec3(0.0, -150.0, 0.0))) * -1.0))) * perlin2(vec2((2.0 * t), 0.0))));
    }
    color = _r2;
    break;
  case 1:
    color = union_color_2(p, normal, occlusion, P, t, light_intensities, camera_origin);
    break;
  }
  return color;
}
float nearest_distance(vec3 p) {
  float t = iTime;
  vec3 p1 = (p - vec3(0.0, 81.0, 0.0));
  vec3 p2 = (p1 * (mat3(1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0) * rotate_y((1.0 * (((sin(t) * 1.500000) + t) * 0.100000)))));
  return union_10(t, p2);
}

vec3 calculate_normal(vec3 p) {
  const vec3 step = vec3(NORMAL_OFFSET, 0.0, 0.0);

  return normalize(vec3(
    nearest_distance(p + step.xyy) - nearest_distance(p - step.xyy),
    nearest_distance(p + step.yxy) - nearest_distance(p - step.yxy),
    nearest_distance(p + step.yyx) - nearest_distance(p - step.yyx)
  ));
}

float calculate_occlusion(vec3 p, vec3 normal) {
  const int step_count = 10;
  const float max_distance = 10.0;
  const float step_size = max_distance / float(step_count);
  float baseline = nearest_distance(p);
  float occlusion = 0.0;
  vec3 step = normal * step_size;
  for (int i = 1; i <= step_count; i++) {
    float expected_distance = baseline + float(i) * step_size;
    float actual_distance = max(nearest_distance(p + float(i) * step), 0.0);
    occlusion += actual_distance / expected_distance;
  }
  occlusion /= float(step_count);
  return clamp(occlusion, 0.0, 1.0);
}

float cast_light(vec3 p, vec3 light, float radius) {
  vec3 direction = normalize(light - p);
  float light_distance = distance(light, p);

  float light_brightness = 1.0 - (light_distance / radius);
  if (light_brightness <= 0.0) {
    return 0.0;
  }

  float in_light = 1.0;
  float sharpness = 16.0;

  float last_distance = 1e20;
  float progress = MINIMUM_HIT_DISTANCE;
  for (int i = 0; i < MAX_STEPS; i++) {
    if (progress > light_distance) {
      return in_light * light_brightness;
    }

    float distance = nearest_distance(p + progress * direction);

    if (distance < MINIMUM_HIT_DISTANCE) {
      return 0.0;
    }

    float intersect_offset = distance * distance / (2.0 * last_distance);
    float intersect_distance = sqrt(distance * distance - intersect_offset * intersect_offset);
    if (distance < last_distance) {
      in_light = min(in_light, sharpness * intersect_distance / max(0.0, progress - intersect_offset));
    }
    progress += distance;
    last_distance = distance;
  }
  return 0.0;
}

vec3 march(vec3 ray_origin, vec3 ray_direction, out int steps) {
  float distance = 0.0;

  for (steps = 0; steps < MAX_STEPS; steps++) {
    vec3 p = ray_origin + distance * ray_direction;
    float nearest = nearest_distance(p);
    if (nearest < MINIMUM_HIT_DISTANCE || distance > MAXIMUM_TRACE_DISTANCE) {
      return p + nearest * ray_direction;
    }

    distance += nearest;
  }
  return ray_origin + distance * ray_direction;
}

vec3 nearest_color(vec3 p, vec3 camera_origin) {
  float t = iTime;
  vec3 normal = calculate_normal(p);
  float occlusion = calculate_occlusion(p, normal);
  vec3 P = p;
  float light_intensities[1];
  light_intensities[0] = cast_light(p + 2.0 * MINIMUM_HIT_DISTANCE * normal, lights[0].position, lights[0].radius);
  vec3 p1 = (p - vec3(0.0, 81.0, 0.0));
  vec3 p2 = (p1 * (mat3(1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0) * rotate_y((1.0 * (((sin(t) * 1.500000) + t) * 0.100000)))));
  return union_color_3(p2, normal, occlusion, P, t, light_intensities, camera_origin);
}

vec3 perspective(float fov, vec2 size, vec2 pos) {
  vec2 xy = pos - size * 0.5;

  float cot_half_fov = tan((90.0 - fov * 0.5) * DEG_TO_RAD);
  float z = min(size.x, size.y) * 0.5 * cot_half_fov;

  return normalize(vec3(xy, -z));
}

mat3 rotate_xy(vec2 angle) {
  vec2 c = cos(angle);
  vec2 s = sin(angle);

  return mat3(
    c.y, 0.0, -s.y,
    s.y * s.x, c.x, c.y * s.x,
    s.y * c.x, -s.x, c.y * c.x
  );
}

//void mainImage( out vec4 fragColor, in vec2 fragCoord )
///////////////////////////////////////////////////////////////////////////////// 
// need to convert this from a void to a function and call it by adding
// a void main(void) { to the end of the shader
// what type of variable will the function return?, it is a color and needs to be a vec4
// change void to vec4 
//void MainImage(out vec4 fragColor, in vec2 fragCoord) 
vec4 mainImage( out vec4 fragColor, in vec2 fragCoord )
{ 
  const float gamma = 2.2;
  vec2 resolution = iResolution.xy;
  vec2 local_coord = gl_FragCoord.xy;

  vec3 camera_origin = vec3(0.0, 50.0 + 20.0 * sin(iTime * 0.25), 420.0);
  vec2 rotation = vec2(-0.125, 0.125 - 0.5 * sin(iTime * 0.25));
  if (iMouse.z > 0.0) {
    vec2 mouse_offset = (iMouse.xy / resolution) - 0.5;
    mouse_offset *= vec2(-1.0, 1.0);
    rotation += mouse_offset;
  }
  mat3 camera_matrix = rotate_xy(rotation.yx);

  camera_origin = camera_matrix * camera_origin + vec3(0.0, -32.0, 0.0);

  int view_type = 1;

  vec3 dir = camera_matrix * perspective(45.0, resolution, gl_FragCoord.xy);

  const vec3 fog_color = vec3(0.15);

  int steps;
  vec3 hit = march(camera_origin, dir, steps);

  vec3 color;
  float depth = distance(camera_origin, hit);
  if (depth >= MAXIMUM_TRACE_DISTANCE) {
    const vec3 light = pow(vec3(69.0, 128.0, 230.0) / vec3(255.0), vec3(gamma));
    const vec3 dark = pow(vec3(40.0, 60.0, 150.0) / vec3(255.0), vec3(gamma));
    color = vec3(mix(dark, light, (local_coord.x + local_coord.y) / (resolution.x + resolution.y)));
  } else {
    color = nearest_color(hit, camera_origin);
  }

  fragColor = vec4(pow(color, vec3(1.0 / gamma)), 1.0);

/////////////////////////////////////////////////////////////////////////////////
//the function needs to return a value. 
//it needs to be a vec4
//we will return the varable fragColor 
// usual place for fragColor = vec4( color, 1.0 ); bring the } down below
return fragColor; 
}

///////////////////////////////////////////////////////////////////////////////// 
void main(void) { // this will be run for every pixel of gl_FragCoord.xy
vec4 vTexCoord = gl_TexCoord[0];
vec4 fragColor = vec4(1.0); // initialize variable fragColor as a vec4 
vec4 cc = mainImage(fragColor, gl_FragCoord.xy); // call function mainImage and assign the return vec4 to cc
gl_FragColor = vec4(cc) * gl_Color; // set the pixel to the value of vec4 cc  and..
//gl_FragColor.a = length(gl_FragColor.rgb);
}

// ..uses the values of any Color: or Opacity:
// clauses (and any Animate clauses applied to these properties) 
// appearing in the Sprite, Quad or other node invoking the shader 
// in the .scn file.

